<?php
/* =========================================================
   WordPress Cleanup, Hardening & Neutral Backend
   Divi 4 / DSGVO-safe / Production Ready
========================================================= */


/* ---------------------------------------------------------
   Clean up Admin Bar (remove personal data & clutter)
--------------------------------------------------------- */
function nxt_remove_admin_bar_links() {
	global $wp_admin_bar;

	$wp_admin_bar->remove_menu('wp-logo');      // WordPress logo
	$wp_admin_bar->remove_menu('new-content');  // + New
	$wp_admin_bar->remove_menu('comments');     // Comments
	$wp_admin_bar->remove_menu('updates');      // Updates
	$wp_admin_bar->remove_menu('my-account');   // User name / email / avatar
}
add_action('wp_before_admin_bar_render', 'nxt_remove_admin_bar_links');


/* ---------------------------------------------------------
   Neutral login screen (no logo, no branding)
--------------------------------------------------------- */
function nxt_login_logo_url() {
	return home_url();
}
add_filter('login_headerurl', 'nxt_login_logo_url');


/* ---------------------------------------------------------
   OPTIONAL: Neutral dashboard widget (no personal data)
--------------------------------------------------------- */
add_action('wp_dashboard_setup', function () {
	wp_add_dashboard_widget(
		'nxt_info_widget',
		'Website Information',
		function () {
			echo '<p>This website is technically maintained and optimized.</p>';
		}
	);
});


/* ---------------------------------------------------------
   Gutenberg backend styles & scripts
--------------------------------------------------------- */
function nxt_add_gutenberg_styles() {

	add_theme_support('editor-styles');

	wp_enqueue_style(
		'nxt_gutenberg_custom_styles',
		get_stylesheet_directory_uri() . '/assets/css/nxt_editor-custom-styles.css',
		[],
		filemtime(get_stylesheet_directory() . '/assets/css/nxt_editor-custom-styles.css')
	);

	wp_register_script(
		'nxt_hack_core_blocks',
		get_stylesheet_directory_uri() . '/assets/js/hack_core_blocks.js',
		[],
		null,
		true
	);
	wp_enqueue_script('nxt_hack_core_blocks');
}
add_action('enqueue_block_editor_assets', 'nxt_add_gutenberg_styles');


/* ---------------------------------------------------------
   Frontend styles & scripts
--------------------------------------------------------- */
function nxt_enqueue_theme_assets() {

	wp_enqueue_style('theme_styles', get_stylesheet_uri());

	wp_register_script(
		'frontend_scripts',
		get_stylesheet_directory_uri() . '/assets/js/frontend_scripts.js',
		[],
		null,
		true
	);
	wp_enqueue_script('frontend_scripts');
}
add_action('wp_enqueue_scripts', 'nxt_enqueue_theme_assets');


/* ---------------------------------------------------------
   Gutenberg feature support
--------------------------------------------------------- */
add_theme_support('custom-spacing');
add_theme_support('border');
add_theme_support('appearance-tools');
add_theme_support('editor-color-palette');


/* ---------------------------------------------------------
   Viewport meta tag
--------------------------------------------------------- */
function nxt_add_viewport_meta() {
	echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">';
}
add_action('wp_head', 'nxt_add_viewport_meta');


/* ---------------------------------------------------------
   Allow SVG uploads (controlled)
--------------------------------------------------------- */
function nxt_allow_svg($mimes) {
	$mimes['svg'] = 'image/svg+xml';
	return $mimes;
}
add_filter('upload_mimes', 'nxt_allow_svg');

function nxt_really_allow_svg($checked, $file, $filename, $mimes) {
	if (!$checked['type']) {
		$wp_filetype = wp_check_filetype($filename, $mimes);
		$ext  = $wp_filetype['ext'];
		$type = $wp_filetype['type'];
		$proper_filename = $filename;

		if ($type && strpos($type, 'image/') === 0 && $ext !== 'svg') {
			$ext = $type = false;
		}

		$checked = compact('ext', 'type', 'proper_filename');
	}
	return $checked;
}
add_filter('wp_check_filetype_and_ext', 'nxt_really_allow_svg', 10, 4);


/* ---------------------------------------------------------
   Redirect non-admin users away from dashboard
--------------------------------------------------------- */
function nxt_redirect_non_admins_from_dashboard() {
	if (
		is_admin() &&
		is_user_logged_in() &&
		!current_user_can('administrator') &&
		!(defined('DOING_AJAX') && DOING_AJAX)
	) {
		wp_safe_redirect(home_url(), 302);
		exit;
	}
}
add_action('init', 'nxt_redirect_non_admins_from_dashboard');


/* ---------------------------------------------------------
   Hide admin bar for non-admins
--------------------------------------------------------- */
function nxt_hide_admin_bar_for_non_admins() {
	if (!current_user_can('administrator')) {
		show_admin_bar(false);
	}
}
add_action('after_setup_theme', 'nxt_hide_admin_bar_for_non_admins');


/* ---------------------------------------------------------
   Sanitize uploaded file names
--------------------------------------------------------- */
function nxt_sanitize_upload_name($filename) {

	$filename = remove_accents($filename);

	$filename = str_replace(
		[' ', '%20', '_'],
		['-', '-', '-'],
		$filename
	);

	$filename = preg_replace('/[^A-Za-z0-9\.\-]/', '', $filename);
	$filename = preg_replace('/\.(?=.*\.)/', '-', $filename);
	$filename = preg_replace('/-+/', '-', $filename);
	$filename = str_replace('-.', '.', $filename);
	$filename = strtolower($filename);

	return $filename;
}
add_filter('sanitize_file_name', 'nxt_sanitize_upload_name', 10);


/* ---------------------------------------------------------
   Register custom block pattern category
--------------------------------------------------------- */
add_action('init', function () {
	register_block_pattern_category(
		'linkmeup',
		[
			'label'       => __('Link Me Up', 'linkmeup'),
			'description' => __('Custom patterns from Link Me Up theme', 'linkmeup'),
		]
	);
});